"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.authConfigHandler = authConfigHandler;
exports.defineAuthRoutes = defineAuthRoutes;
exports.logoutHandler = logoutHandler;
var _constants = require("../../../utils/constants");
var _AuthManager = require("./AuthManager");
var _routes = require("./types/common/routes");
/*
 *    Copyright 2021 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

function defineAuthRoutes({
  kibanaCore,
  authManager,
  searchGuardBackend,
  configService
}) {
  const router = kibanaCore.http.createRouter();
  const httpResources = kibanaCore.http.resources;
  (0, _routes.customError)({
    httpResources
  });
  router.post({
    path: `${_constants.API_ROOT}/auth/logout`,
    validate: false
  }, logoutHandler({
    authManager
  }));
  router.get({
    path: `${_constants.API_ROOT}/auth/config`,
    validate: false,
    options: {
      authRequired: false
    }
  }, authConfigHandler({
    authManager,
    searchGuardBackend,
    configService,
    kibanaCore
  }));
  httpResources.register({
    path: `/auth/captureurlfragment`,
    options: {
      authRequired: false
    },
    validate: false
  }, (context, request, response) => {
    const loginHandler = request.url.searchParams.get('loginHandler');
    if (!loginHandler.match(/\/[a-zA-Z0-9\/]+/)) {
      return response.renderHtml({
        body: 'Bad Request'
      });
    }
    const authTypeId = request.url.searchParams.get('authTypeId');
    if (authTypeId && !authTypeId.match(/[a-zA-Z0-9]+/)) {
      return response.renderHtml({
        body: 'Bad Request'
      });
    }
    return response.renderHtml({
      body: `<html><head><script src="captureurlfragment.js"></script></head><body></body></html>`
    });
  });
  httpResources.register({
    path: `/auth/captureurlfragment.js`,
    options: {
      authRequired: false
    },
    validate: false
  }, (context, request, response) => {
    return response.renderJs({
      body: `
          let searchParams = new URLSearchParams(window.location.search);
          let loginHandler = searchParams.get('loginHandler');
          let authTypeId = searchParams.get('authTypeId');
          let nextUrl = searchParams.get('nextUrl');
          if (!nextUrl) nextUrl = "/";
          if (window.location.hash) nextUrl += window.location.hash;
          window.location = loginHandler + "?authTypeId=" + encodeURIComponent(authTypeId) + "&nextUrl=" + encodeURIComponent(nextUrl);
          `
    });
  });
}
function logoutHandler({
  authManager
}) {
  return async function (context, request, response) {
    return authManager.logout({
      context,
      request,
      response
    });
  };
}
function authConfigHandler({
  authManager,
  searchGuardBackend,
  configService,
  kibanaCore
}) {
  return async function (context, request, response) {
    const authConfig = await searchGuardBackend.getAuthConfig();
    const backendMethodToFrontendMethod = {
      basic: _AuthManager.AUTH_TYPE_NAMES.BASIC,
      oidc: _AuthManager.AUTH_TYPE_NAMES.OIDC,
      openid: _AuthManager.AUTH_TYPE_NAMES.OIDC,
      saml: _AuthManager.AUTH_TYPE_NAMES.SAML,
      jwt: _AuthManager.AUTH_TYPE_NAMES.JWT,
      link: _AuthManager.AUTH_TYPE_NAMES.JWT
    };
    const authTypes = authConfig.auth_methods.filter(config => authManager.authInstances[backendMethodToFrontendMethod[config.method]]).map(config => {
      const authInstance = authManager.authInstances[backendMethodToFrontendMethod[config.method]];
      if (authInstance) {
        let loginURL = authInstance.loginURL;
        if (config.id) {
          // All loginURLs are relative
          loginURL = kibanaCore.http.basePath.get() + authInstance.loginURL + "?authTypeId=" + encodeURIComponent(config.id);
        }

        // For example, we don't have a loginURL flow for JWT. Instead,
        // we can use the sso_location directly
        if (!authInstance.loginURL && config.sso_location) {
          loginURL = config.sso_location;
        }
        return {
          id: config.id,
          type: backendMethodToFrontendMethod[config.method],
          title: config.label,
          loginURL,
          message_body: config.message_body,
          message_title: config.message_title,
          details: config.details,
          unavailable: config.unavailable
        };
      }
    });
    return response.ok({
      body: {
        authTypes,
        loginPage: authConfig.login_page
      }
    });
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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